/**************************************************************************************

   Copyright (c) Hilscher GmbH. All Rights Reserved.

 **************************************************************************************

   Filename:
    $Id: DeviceHandler.cpp 13525 2020-01-20 12:32:36Z Robert $
   Last Modification:
    $Author: Robert $
    $Date: 2020-01-20 13:32:36 +0100 (Mo, 20 Jan 2020) $
    $Revision: 13525 $

   Targets:
     Win32/ANSI   : yes
     Win32/Unicode: yes (define _UNICODE)
     WinCE        : no

   Description:
     Implementation of the "Device Handler" Class, which handles all physical layers

   Changes:

     Version   Date        Author   Description
     ----------------------------------------------------------------------------------
     13        17.01.20    RM       BrowseDevices(), don't check endpoints if EstablishConnection() failes at all,
                                    preventing multiple tries to establish connections to underlying endpoints.
     12        03.04.19    AM       Added support for xSysdeviceResetEx().
                                    Reverted changes made in NXTDLL-57.
     11        18.04.17    FF       Changed the algorithm to find the current device
                                      - was the problem with USB-devices during
                                        connecting and disconnecting devices
                                      - NXTDLL-57
     10        01.07.14    SS       Improved driver startup and board enumeration performance
     9         20.03.14    SD       - Bugfix: Deadlock situation in case of failing attempt of
                                      connection establishment during disconnection (concurrent
                                      call of InterfaceDisconnect()). Since InterfaceDisconnect()
                                      is called directly from another context (InterfaceMonitorThread())
                                      it still misses appropriate locking.
                                      Added missing "connect-lock" request in InterfaceDisconnect()
                                      Removed redundant locking within ProcessNotification() (connect
                                      already locked in subsequent function calls)
     8         07.03.14    SD       Freeing the physical layer before stopping the monitoring
                                    threads may lead to access violation (DeviceHandler destructor)
     7         07.10.11    SS       Reduce the timeout of RPC involved in connection establishment
     6         31.03.11    SS       - Access to invalid data layer crashes application
                                    - Remove of physical layer now done by Destructor
                                    - Unnecessary functions RemoveAllLayers(),
                                      LockChannelAccess(), UnlockChannelAccess() removed
                                    - Disconnect() and CheckInterfaceDisconnect()
                                      merged into InterfaceDisconnect()
     5         12.10.10    MT       Bugfix: Reconnect Event was not closed after reset has completed
     4         06.10.10    SS       Bugfix: Unkown data layers not handled during
                                    connection establishment
                                    Bugfix: Connection failed if query server command
                                    is not supported on device side (only cifX API Marshaller)

     3         08.04.10    SS       Bugfix: EstablishConnection() Invalid access to
                                    datalayer because device flag is set available before
                                    assigning default datalayer to endpoint
                                    Bugfix: Disconnect()/ReleaseConnection() Invalid access
                                    to device because transport layer is deinitialized before
                                    device flag is set unavailable
                                    Bugfix: InterfaceMonitorThreadFunc() may deadlock

     2         03.03.09    MT       Structural change to allow placing physical drivers
                                    in a dll

     1         25.09.08    PL       initial version

**************************************************************************************/

/////////////////////////////////////////////////////////////////////////////
/// \file DeviceHandler.cpp
/// Implementation of CDeviceHandler class
/////////////////////////////////////////////////////////////////////////////

#include "StdAfx.h"
#include "DeviceHandler.h"
#include "ConnectorAPI.h"
#include "PhysicalInterface.h"
#include "Marshaller.h"
#include "rcXPacket.h"
#include "TransportLayer.h"
#include "HelperFunctions.h"

#include "cifXErrors.h"

#include <algorithm>

#if 0
  /* Visual Leak Detector (Version 1.9d) - Import Library Header */
  #pragma comment(lib, "C:\\Programme\\Visual Leak Detector\\lib\\vld.lib")
  #pragma comment(linker, "/include:__imp_?vld@@3VVisualLeakDetector@@A")
#endif

///////////////////////////////////////////////////////////////////////////////////////////
/// Constructor of CDeviceHandler class
///////////////////////////////////////////////////////////////////////////////////////////
CDeviceHandler::CDeviceHandler(void)
: m_hInterfaceNotifyThread(NULL)
, m_hInterfaceNotifySemaphore(NULL)
, m_fInterfaceNotifyStop(false)
, m_hInterfaceMonitorThread(NULL)
, m_hInterfaceMonitorStop(NULL)

{
  InitializeCriticalSection(&m_tcsNotificationLock);
  InitializeCriticalSection(&m_tcsInterfaceLock);
  InitializeCriticalSection(&m_tcsInterfaceMonitorQueue);
}

///////////////////////////////////////////////////////////////////////////////////////////
/// Denstructor of CDeviceHandler class
///////////////////////////////////////////////////////////////////////////////////////////
CDeviceHandler::~CDeviceHandler(void)
{
  /* stop all running threads, before removing the physical layers */
  Deinit();

  /* remove all registered physical layers */
  while ( 0 != m_cvLayers.size())
  {
    CPhysicalLayer* pcLayer = m_cvLayers[0];
    // Search given layer in layer vector and clean all correlations
    for(size_t iIdx = 0; iIdx < m_cvLayers.size(); ++iIdx)
    {
      if(m_cvLayers[iIdx] == pcLayer)
      {
        // DeInitialize the connector layer
        pcLayer->Deinit();

        EnterCriticalSection(&m_tcsInterfaceLock);

        /* Remove all interfaces, endpoints, etc. */
        INTERFACE_MAP::iterator iterIntf = m_cmInterfaces.begin();
        while (m_cmInterfaces.end() != iterIntf)
        {
          INTERFACE_DATA_T& tInterface = iterIntf->second;
          if (tInterface.pcInterface->GetPhysicalLayer() == pcLayer)
          {
            CTransportLayer* pcTransportLayer = tInterface.pcInterface->GetTransportLayer();
            /* This interface belongs to our physical layer */
            ReleaseConnection(tInterface);
            delete pcTransportLayer;
            delete tInterface.pcInterface;
            m_cmInterfaces.erase(iterIntf);
            iterIntf = m_cmInterfaces.begin();
          } else
          {
            /* As this interface belongs to another physical layer,
               handle next interface in map */
            iterIntf++;
          }
        }

        LeaveCriticalSection(&m_tcsInterfaceLock);

        // Remove layer from map
        m_cvLayers.erase(m_cvLayers.begin() + iIdx);
        delete pcLayer;
        break;
      }
    }
  }

  DeleteCriticalSection(&m_tcsNotificationLock);
  DeleteCriticalSection(&m_tcsInterfaceLock);
  DeleteCriticalSection(&m_tcsInterfaceMonitorQueue);
}

///////////////////////////////////////////////////////////////////////////////////////////
/// Device handler initialization
///   \return CIFX_NO_ERROR on success
///////////////////////////////////////////////////////////////////////////////////////////
int32_t CDeviceHandler::Init(void)
{
  m_fInterfaceNotifyStop = false;

  int32_t lRet = CIFX_DRV_INIT_ERROR;

  if(NULL == (m_hInterfaceNotifySemaphore = ::CreateSemaphore(NULL, 0, MAXLONG, NULL)))
  {
    lRet = CIFX_DRV_INIT_ERROR;

  } else if(NULL == (m_hInterfaceNotifyThread = ::CreateThread(NULL,
                                                               0,
                                                               InterfaceNotifyThread,
                                                               this,
                                                               0,
                                                               NULL)))
  {
    lRet = CIFX_DRV_INIT_ERROR;

  } else if( NULL == (m_hInterfaceMonitorStop = ::CreateEvent(NULL, FALSE, FALSE, NULL)) )
  {
    lRet = CIFX_DRV_INIT_ERROR;

  } else if(NULL == (m_hInterfaceMonitorThread = ::CreateThread(NULL,
                                                                0,
                                                                InterfaceMonitorThread,
                                                                this,
                                                                0,
                                                                NULL)))
  {
    lRet = CIFX_DRV_INIT_ERROR;

  } else
  {
    lRet = CIFX_NO_ERROR;
  }

  if(CIFX_NO_ERROR != lRet)
  {
    Deinit();
  }

  return lRet;
}

///////////////////////////////////////////////////////////////////////////////////////////
/// Device handler deinitialization
///////////////////////////////////////////////////////////////////////////////////////////
void CDeviceHandler::Deinit(void)
{
  if(NULL != m_hInterfaceMonitorThread)
  {
    SetEvent(m_hInterfaceMonitorStop);
    if(::WaitForSingleObject(m_hInterfaceMonitorThread, 10000) != WAIT_OBJECT_0)
      ::TerminateThread(m_hInterfaceMonitorThread, MAXDWORD);

    ::CloseHandle(m_hInterfaceMonitorThread);
    m_hInterfaceMonitorThread = NULL;
  }

  if(NULL != m_hInterfaceMonitorStop)
  {
    ::CloseHandle(m_hInterfaceMonitorStop);
    m_hInterfaceMonitorStop = NULL;
  }

  if(NULL != m_hInterfaceNotifyThread)
  {
    m_fInterfaceNotifyStop = true;
    ::ReleaseSemaphore(m_hInterfaceNotifySemaphore, 1, NULL);

    if(::WaitForSingleObject(m_hInterfaceNotifyThread, 10000) != WAIT_OBJECT_0)
      ::TerminateThread(m_hInterfaceNotifyThread, MAXDWORD);

    ::CloseHandle(m_hInterfaceNotifyThread);
    m_hInterfaceNotifyThread = NULL;
  }

  if(NULL != m_hInterfaceNotifySemaphore)
  {
    ::CloseHandle(m_hInterfaceNotifySemaphore);
    m_hInterfaceNotifySemaphore = NULL;
  }
}

///////////////////////////////////////////////////////////////////////////////////////////
/// Interface notification thread
///   \param pvParam User data
///   \return 0 always
///////////////////////////////////////////////////////////////////////////////////////////
DWORD CDeviceHandler::InterfaceNotifyThread(void* pvParam)
{
  CDeviceHandler* pcDeviceHandler = reinterpret_cast<CDeviceHandler*>(pvParam);
  bool            fRunning        = true;

  while(fRunning)
  {
    DWORD dwWait = ::WaitForSingleObject(pcDeviceHandler->m_hInterfaceNotifySemaphore, INFINITE);

    switch(dwWait)
    {
    case WAIT_OBJECT_0:
      if(pcDeviceHandler->m_fInterfaceNotifyStop)
      {
        fRunning = false;
      } else
      {
        int32_t lError = CIFX_NO_ERROR;
        EnterCriticalSection(&pcDeviceHandler->m_tcsNotificationLock);

        NOTIFICATION_DATA_T tData = pcDeviceHandler->m_cvInterfaceNotifications.front();
        pcDeviceHandler->m_cvInterfaceNotifications.pop_front();

        LeaveCriticalSection(&pcDeviceHandler->m_tcsNotificationLock);

        /* Handle Interface notification */
        switch(tData.eNotify)
        {
          case eATTACHED:
            lError = pcDeviceHandler->EstablishConnection(*tData.ptInterface);
          break;

          case eDETACHED:
            pcDeviceHandler->InterfaceDisconnect(*tData.ptInterface);
          break;

          case eDELETED:
            pcDeviceHandler->ReleaseConnection(*tData.ptInterface);
          break;

          default:
            _ASSERT(false);
          break;
        }

        TRACE("%s: %s (Error 0x%x)\n", tData.ptInterface->pcInterface->GetInterfaceName().c_str(), tData.eNotify==eATTACHED?"ATTACHED":"DETACHED", lError);
      }
      break;

    default:
      _ASSERT(false);
      break;
    }

  }

  return 0;
}

///////////////////////////////////////////////////////////////////////////////////////////
/// Add a physical layer to the device handler
///   \param  pcLayer Reference of the physical layer object
///   \return CIFX_NO_ERROR on Succes
///////////////////////////////////////////////////////////////////////////////////////////
int32_t CDeviceHandler::AddLayer(CPhysicalLayer* pcLayer)
{
  m_cvLayers.push_back(pcLayer);
  pcLayer->SetDeviceHandler(this);

  return pcLayer->Init();
}

///////////////////////////////////////////////////////////////////////////////////////////
/// Handle interface notification callbacks (interface removed / attached)
///  \param szInterface Interface name
///  \param eNotify     Notification information structure
///  \param pcLayer     Physical layer object
///////////////////////////////////////////////////////////////////////////////////////////
void CDeviceHandler::HandleInterfaceNotification( std::string&                  szInterface,
                                                  NETX_INTERFACE_NOTIFICATION_E eNotify,
                                                  CPhysicalLayer*               pcLayer)
{
  // Check interface notification event was initialized
  NOTIFICATION_DATA_T tNotifyData;
  tNotifyData.eNotify = eNotify;

  EnterCriticalSection(&m_tcsInterfaceLock);

  /* Search if we already know this Interface */
  INTERFACE_MAP::iterator iter = m_cmInterfaces.find(szInterface);

  if(iter == m_cmInterfaces.end())
  {
    /* We need to create a new interface and add it to list */
    CPhysicalInterface* pcInterface = pcLayer->CreateInterface(szInterface);
    if (NULL != pcInterface)
    {
      INTERFACE_DATA_T tInterface;
      CTransportLayer* pcTransportLayer = new CTransportLayer(pcInterface, this);

      tInterface.eState           = eOFFLINE;
      tInterface.pcInterface      = pcInterface;
      tInterface.pcInterface->SetTransportLayer(pcTransportLayer);
      m_cmInterfaces[szInterface] = tInterface;
      tNotifyData.ptInterface     = &m_cmInterfaces[szInterface];
    }
  } else
  {
    tNotifyData.ptInterface = &iter->second;
  }

  LeaveCriticalSection(&m_tcsInterfaceLock);

  // Lock notification map
  EnterCriticalSection(&m_tcsNotificationLock);

  m_cvInterfaceNotifications.push_back(tNotifyData);

  // UnLock notification map
  LeaveCriticalSection(&m_tcsNotificationLock);

  ::ReleaseSemaphore(m_hInterfaceNotifySemaphore, 1, NULL);
}

///////////////////////////////////////////////////////////////////////////////////////////
/// Establish a new connection by adding it to the device handler
///   \param tInterface Interface data
///   \return CIFX_NO_ERROR on success
///////////////////////////////////////////////////////////////////////////////////////////
int32_t CDeviceHandler::EstablishConnection(INTERFACE_DATA_T& tInterface)
{
  int32_t lRet = CIFX_NO_ERROR;
  CTransportLayer* pcTransportLayer = tInterface.pcInterface->GetTransportLayer();

  ScheduleInterfaceMonitor(tInterface.pcInterface->GetInterfaceName(), eRESET);

  tInterface.pcInterface->Lock();

  /* Initialize all endpoints */
  if (tInterface.eState == eONLINE)
  {
    /* Already connected */
  } else if(CIFX_NO_ERROR != (lRet = tInterface.pcInterface->Start()))
  {
    /* Error starting interface */
    TRACE("%s: Initialization of interface failed (lRet = 0x%08x)!",
          tInterface.pcInterface->GetInterfaceName().c_str(), lRet);
    tInterface.eState = eOFFLINE;

  } else if(CIFX_NO_ERROR != (lRet = pcTransportLayer->Init()))
  {
    /* Error Initializing layer */
    TRACE("%s: Initialization of transport layer failed (lRet = 0x%08x)!",
          tInterface.pcInterface->GetInterfaceName().c_str(), lRet);
    tInterface.eState = eOFFLINE;

  } else
  {
    std::vector<uint16_t> cvCheckTypes;

    if(CIFX_NO_ERROR != (pcTransportLayer->QueryServerFeatures(CIFX_INIT_TIMEOUT)))
    {
      /* No Server features available, so we need to check all packet types */
      TRACE("%s: QueryServerFeatures not supported!",
            tInterface.pcInterface->GetInterfaceName().c_str());
      cvCheckTypes.push_back(HIL_TRANSPORT_TYPE_RCX_PACKET);
      cvCheckTypes.push_back(HIL_TRANSPORT_TYPE_MARSHALLER);

    } else
    {
      CTransportLayer::TARGET_SERVER_INFO_T* ptServerInfo = pcTransportLayer->GetServerInfo();
      cvCheckTypes = ptServerInfo->cvSupportedDataTypes;
    }

    tInterface.eState = eONLINE;

    /* TODO: Check if Device supports HIL_TRANSPORT_QUERYDEVICE for multiple endpoints */

    CEndpoint* pcEndpoint     = NULL;
    bool       fReopenChannel = false;

    if(tInterface.cvEndpoints.size() > 0)
    {
      /* This is a reconnect */
      fReopenChannel = true;
      pcEndpoint     = tInterface.cvEndpoints[0];
    } else
    {
      pcEndpoint    = new CEndpoint(tInterface.pcInterface);
    }

    /* Sort by datatype, so the default layer selection works */
    sort(cvCheckTypes.begin(), cvCheckTypes.end());

    unsigned int uiUnkownTypes = 0;
    /* Server Features are available, so we can use the server information */
    for(size_t uiIdx = 0; uiIdx < cvCheckTypes.size(); uiIdx++)
    {
      uint16_t usDataType = cvCheckTypes[uiIdx];

      switch(usDataType)
      {
        case HIL_TRANSPORT_TYPE_RCX_PACKET:
        {
          CrcXPacket* pcRCXPacket = NULL;

          /* Search if we need to create a new data layer */
          if (pcEndpoint->m_cmDataLayers.end() == pcEndpoint->m_cmDataLayers.find(usDataType))
            pcRCXPacket = new CrcXPacket(pcTransportLayer, pcEndpoint);
          else
            pcRCXPacket = (CrcXPacket*)pcEndpoint->m_cmDataLayers[usDataType];

          if(CIFX_NO_ERROR != pcRCXPacket->Init())
          {
            /* Device does not support packets */
            pcRCXPacket->Deinit();
          }
        }
        break;

        case HIL_TRANSPORT_TYPE_MARSHALLER:
        {
          CMarshaller* pcMarshaller = NULL;

          /* Search if we need to create a new data layer */
          if (pcEndpoint->m_cmDataLayers.end() == pcEndpoint->m_cmDataLayers.find(usDataType))
            pcMarshaller = new CMarshaller(pcTransportLayer, pcEndpoint);
          else
            pcMarshaller = (CMarshaller*)pcEndpoint->m_cmDataLayers[usDataType];

          if(CIFX_NO_ERROR != pcMarshaller->Init())
          {
            /* Device does not support packets */
            pcMarshaller->Deinit();
          }
        }
        break;

        default:
          uiUnkownTypes++;
        break;
      }
    }

    if (uiUnkownTypes == cvCheckTypes.size())
    {
      /* The data types provided by the query server request are unknown.
        Connection to the device is not possible */
      TRACE("%s: The data layers provided by the device are not supported!",
            tInterface.pcInterface->GetInterfaceName().c_str());

      /* Deletet Endpoint will remove the data layer, devices and its channels */
      if(!fReopenChannel)
        delete pcEndpoint;

      lRet = CIFX_TRANSPORT_UNKNOWN_DATALAYER;

    } else if ( (!pcEndpoint->m_pcDefaultDataLayer) || (!pcEndpoint->m_pcDefaultDataLayer->ConnectionValid()) )
    {
      /* Initialization of data layers failed */
      TRACE("%s: Initialization of data layers failed!",
            tInterface.pcInterface->GetInterfaceName().c_str());

      /* Deletet Endpoint will remove the data layer, devices and its channels */
      if(!fReopenChannel)
        delete pcEndpoint;

      lRet = CIFX_DRV_INIT_ERROR;

    } else if(fReopenChannel)
    {
      /* Try to reopen the device and the coresponding channels */
      uint32_t    ulDevice = 0;
      CDevice* pcDevice = NULL;

      while(NULL != (pcDevice = pcEndpoint->GetDeviceObject(ulDevice++)))
      {
        // check the device
        if(!pcDevice->IsAvailable())
        {
          /* This device does not exist anymore */
          continue;
        }

        // Device is available, open previously opened channels
        CChannel* pcChannel = pcDevice->GetChannelObject(HIL_SYSTEM_CHANNEL);
        if ( (pcChannel != NULL) && (0 < pcChannel->m_lOpenRefCount) )
        {
          pcEndpoint->m_pcDefaultDataLayer->xSysdeviceOpen(this, (char*)pcDevice->m_tBoardInfo.abBoardName,
                                                           &pcChannel->m_hChannel);
        }

        uint32_t ulChannel = 0;

        while(NULL != (pcChannel = pcDevice->GetChannelObject(ulChannel++)))
        {
          if (!pcChannel->IsAvailable())
          {
            /* Channel is not available anymore, so skip rest of channels */
            break;
          } else if (0 == pcChannel->m_lOpenRefCount)
          {
            /* Channel was not opened before reconnect, so continue with next channel */
            continue;
          }

          pcEndpoint->m_pcDefaultDataLayer->xChannelOpen(this, (char*)pcDevice->m_tBoardInfo.abBoardName,
                                                         ulChannel-1, &pcChannel->m_hChannel);
        }
      }
    } else
    {
      /* Reopen Channels means we already had this endpoint, so don't add it to the list
        again, as it is already there */
      tInterface.cvEndpoints.push_back(pcEndpoint);
    }

    if (CIFX_NO_ERROR != lRet)
    {
      /* Failed to establish connection */
      InterfaceDisconnect(tInterface);
    } else
    {
      ScheduleInterfaceMonitor(tInterface.pcInterface->GetInterfaceName(), eDISCONNECT);
      if(tInterface.pcInterface->m_hReconnect != NULL)
        ::SetEvent(tInterface.pcInterface->m_hReconnect);
    }
  }

  tInterface.pcInterface->Unlock();

  return lRet;
}

///////////////////////////////////////////////////////////////////////////////////////////
/// Release the connection on an interface
///   \\param tInterface Interface data structure
///////////////////////////////////////////////////////////////////////////////////////////
void CDeviceHandler::ReleaseConnection(INTERFACE_DATA_T& tInterface)
{
  tInterface.pcInterface->Lock();

  // Stop physical interface
  ScheduleInterfaceMonitor(tInterface.pcInterface->GetInterfaceName(), eRESET);
  InterfaceDisconnect(tInterface);
  while(tInterface.cvEndpoints.size() > 0)
  {
    CEndpoint* pcEndpoint = *tInterface.cvEndpoints.begin();

    /* Deletet Endpoint will remove the data layer, devices and its channels */
    delete pcEndpoint;

    tInterface.cvEndpoints.erase(tInterface.cvEndpoints.begin());
  }
  tInterface.pcInterface->Unlock();
}

///////////////////////////////////////////////////////////////////////////////////////////
/// Get the number of devices over all interfaces
///   \return Number of devices on all interfaces
///////////////////////////////////////////////////////////////////////////////////////////
uint32_t CDeviceHandler::GetDeviceCount(void)
{
  uint32_t ulRet = 0;

  EnterCriticalSection(&m_tcsInterfaceLock);

  INTERFACE_MAP::iterator iter = m_cmInterfaces.begin();

  while(iter != m_cmInterfaces.end())
  {
    for(size_t uiEndpoint = 0; uiEndpoint < iter->second.cvEndpoints.size(); uiEndpoint++)
    {
      CEndpoint* pcEndpoint = iter->second.cvEndpoints[uiEndpoint];

      ulRet += pcEndpoint->GetDeviceCount();
    }
    iter++;
  }

  LeaveCriticalSection(&m_tcsInterfaceLock);

  return ulRet;
}

///////////////////////////////////////////////////////////////////////////////////////////
/// Search a device by a given board number
///   \param  ulBoard     Board number to search for
///   \param  fOpenDevice TRUE if the interface should be started
///   \return CDevice* Pointer of the device object on success
///////////////////////////////////////////////////////////////////////////////////////////
CDevice* CDeviceHandler::GetDevice(uint32_t ulBoard, bool fOpenDevice /*= false */)
{
  CDevice* pcRet = NULL;
  uint32_t ulCurrentBoard = 0;

  EnterCriticalSection(&m_tcsInterfaceLock);

  INTERFACE_MAP::iterator iter = m_cmInterfaces.begin();

  while (iter != m_cmInterfaces.end())
  {
    INTERFACE_DATA_T& tInterface = (iter++)->second;

    tInterface.pcInterface->Lock();
    LeaveCriticalSection(&m_tcsInterfaceLock);

    if (eOFFLINE == tInterface.eState)
    {
      int32_t lRet = CIFX_TRANSPORT_CONNECT;
      if ((fOpenDevice) && (ulCurrentBoard >= ulBoard))
      {
        /* Connection establishment failed previously. Try again to establish a connection before finding the board */
        lRet = EstablishConnection(tInterface);
      }
      if (CIFX_NO_ERROR != lRet)
      {
        tInterface.pcInterface->Unlock();
        EnterCriticalSection(&m_tcsInterfaceLock);
        continue;
      }
    }
    tInterface.pcInterface->Unlock();

    for (size_t uiEndpoint = 0; uiEndpoint < tInterface.cvEndpoints.size(); uiEndpoint++)
    {
      CEndpoint* pcEndpoint = tInterface.cvEndpoints[uiEndpoint];

      if (pcEndpoint->m_pcDefaultDataLayer != NULL)
      {
        uint32_t ulDeviceCount = pcEndpoint->GetDeviceCount();

        if ((ulCurrentBoard + ulDeviceCount) == ulBoard + 1)
        {
          /* This endpoint holds the board we are interested in */
          uint32_t ulEndpointSpecificBoardNr = ulBoard - ulCurrentBoard;

          pcRet = pcEndpoint->GetDeviceObject(ulEndpointSpecificBoardNr);

          iter = m_cmInterfaces.end();
          break;
        } else
        {
          ulCurrentBoard += ulDeviceCount;
        }
      }
    }
    EnterCriticalSection(&m_tcsInterfaceLock);
  }

  LeaveCriticalSection(&m_tcsInterfaceLock);

  return pcRet;
}

///////////////////////////////////////////////////////////////////////////////////////////
/// Search a device by a given board name
///   \param  szBoard  Board name as a string
///   \return CDevice* Pointer of the device object on success
///////////////////////////////////////////////////////////////////////////////////////////
CDevice* CDeviceHandler::GetDevice(char* szBoard)
{
  CDevice*      pcRet          = NULL;
  std::string   cInterfaceName;
  std::string   cBoardName;

  /* Splitup I/F / Board */
  if(SplitDeviceName(szBoard, cInterfaceName, cBoardName))
  {
    EnterCriticalSection(&m_tcsInterfaceLock);

    INTERFACE_MAP::iterator iter = m_cmInterfaces.find(cInterfaceName);

    if(iter != m_cmInterfaces.end())
    {
      /* We've found the interface the board is connected to */
      INTERFACE_DATA_T& tInterface = iter->second;
      int32_t           lRet       = CIFX_NO_ERROR;

      tInterface.pcInterface->Lock();
      LeaveCriticalSection(&m_tcsInterfaceLock);

      if(eONLINE != tInterface.eState)
      {
        /* Interface offline or paused. Try to establish a connection before finding the board */
        lRet = EstablishConnection(tInterface);
      }
      tInterface.pcInterface->Unlock();

      if (CIFX_NO_ERROR == lRet)
      {
        for(size_t uiEndpoint = 0; uiEndpoint < tInterface.cvEndpoints.size(); uiEndpoint++)
        {
          CEndpoint* pcEndpoint = tInterface.cvEndpoints[uiEndpoint];

          if(NULL != (pcRet = pcEndpoint->GetDeviceObject( cBoardName.c_str())))
          {
            /* We've found our board */
            break;
          }
        }
      }
    } else
    {
      LeaveCriticalSection(&m_tcsInterfaceLock);
    }
  }

  return pcRet;
}

///////////////////////////////////////////////////////////////////////////////////////////
/// Open a device connection
///  \param  szBoard      Name of the board
///  \param  ulChannel    Channel number on the board
///  \param  phChannel    Buffer to return the correponding handle
///  \return  CIFX_NO_ERROR on success
///////////////////////////////////////////////////////////////////////////////////////////
int32_t CDeviceHandler::OpenDeviceConnection(char* szBoard, uint32_t ulChannel, CIFXHANDLE* phChannel)
{
  int32_t     lRet     = CIFX_INVALID_BOARD;

  CDevice* pcDevice = GetDevice(szBoard);

  if(NULL != pcDevice)
  {
    CChannel* pcChannel = pcDevice->GetChannelObject(ulChannel);

    if(NULL != pcChannel)
    {
      pcDevice->m_pcEndpoint->m_pcInterface->Lock();

      // Check if the device is connected
      if( pcDevice->IsAvailable())
      {
        if(pcChannel->m_lOpenRefCount == 0)
        {
          /* We are the first to open the device */
          if(ulChannel == HIL_SYSTEM_CHANNEL)
          {
            lRet = pcDevice->m_pcEndpoint->m_pcDefaultDataLayer->xSysdeviceOpen(this,
                                                                                (char*)pcDevice->m_tBoardInfo.abBoardName,
                                                                                &pcChannel->m_hChannel);
          } else
          {
            lRet = pcDevice->m_pcEndpoint->m_pcDefaultDataLayer->xChannelOpen(this,
                                                                              (char*)pcDevice->m_tBoardInfo.abBoardName,
                                                                              ulChannel,
                                                                              &pcChannel->m_hChannel);
          }
        } else
        {
          /* Channel is already open */
          lRet = CIFX_NO_ERROR;
        }

        if(CIFX_NO_ERROR == lRet)
        {
          pcChannel->m_lOpenRefCount++;
          if (NULL != phChannel)
            *phChannel = pcChannel;
        }
      }

      pcDevice->m_pcEndpoint->m_pcInterface->Unlock();
    }
  }

  return lRet;
}

///////////////////////////////////////////////////////////////////////////////////////////
/// Close a device connection
///   \param  hDevCon  Device hanlde
///   \return CIFX_NO_ERROR on success
///////////////////////////////////////////////////////////////////////////////////////////
int32_t CDeviceHandler::CloseDeviceConnection(HANDLE hDevCon)
{
  int32_t      lRet      = CIFX_INVALID_BOARD;
  CChannel* pcChannel = reinterpret_cast<CChannel*>(hDevCon);;

  if(NULL != pcChannel)
  {
    pcChannel->m_pcDevice->m_pcEndpoint->m_pcInterface->Lock();

    if(pcChannel->m_lOpenRefCount == 0)
    {
      /* Device was not opened before */
      lRet = CIFX_DRV_NOT_OPENED;

    } else if(--pcChannel->m_lOpenRefCount == 0)
    {
      CDevice* pcDevice = pcChannel->m_pcDevice;

      if( pcDevice->IsAvailable())
      {
        /* We are the last, so close device */
        if(pcChannel->m_ulChannelNumber == HIL_SYSTEM_CHANNEL)
        {
          lRet = pcDevice->m_pcEndpoint->m_pcDefaultDataLayer->xSysdeviceClose(pcChannel->m_hChannel);
        } else
        {
          lRet = pcDevice->m_pcEndpoint->m_pcDefaultDataLayer->xChannelClose(pcChannel->m_hChannel);
        }
      }

      pcChannel->m_hChannel = NULL;

      ScheduleInterfaceMonitor(pcDevice->m_pcEndpoint->m_pcInterface->GetInterfaceName(), eDISCONNECT);

    } else
    {
      /* Channel is still open by another instance */
      lRet = CIFX_NO_ERROR;
    }

    pcChannel->m_pcDevice->m_pcEndpoint->m_pcInterface->Unlock();
  }

  return lRet;
}

///////////////////////////////////////////////////////////////////////////////////////////
/// Enuerate available devices
///   \param  ulBoard     Board number
///   \param  ptBoardInfo Buffer for the device information
///   \return CIFX_NO_ERROR on success
///////////////////////////////////////////////////////////////////////////////////////////
int32_t CDeviceHandler::EnumerateDevice(uint32_t ulBoard, BOARD_INFORMATION* ptBoardInfo)
{
  int32_t     lRet     = CIFX_NO_MORE_ENTRIES;

  CDevice* pcDevice = GetDevice(ulBoard, true);

  if(NULL != pcDevice)
  {
    std::string szInterfaceName;
    pcDevice->m_pcEndpoint->m_pcInterface->GetInterfaceName(szInterfaceName);

    // Copy the board information
    *ptBoardInfo = pcDevice->m_tBoardInfo;

    // Return the complete name for the user
    CreateInterfaceName(szInterfaceName,
                        pcDevice->m_tBoardInfo.abBoardName,
                        sizeof(pcDevice->m_tBoardInfo.abBoardName)+sizeof(pcDevice->m_tBoardInfo.abBoardAlias),
                        ptBoardInfo->abBoardName);

    lRet = CIFX_NO_ERROR;
  }

  return lRet;
}

///////////////////////////////////////////////////////////////////////////////////////////
/// Enuerate available devices
///   \param  cvBoardList List containing complete browse result
///   \param  pfnCallback Callback to process browsed devices
///   \param  pvUser      User parameter for browse callback
///   \return CIFX_NO_ERROR on success
///////////////////////////////////////////////////////////////////////////////////////////
int32_t CDeviceHandler::BrowseDevices(std::vector<BOARD_INFORMATION>& cvBoardList, PFN_NXAPI_BROWSE_CALLBACK pfnCallback /*= NULL */, void* pvUser /*=NULL */)
{
  int32_t          lRet   = CIFX_NO_ERROR;
  uint32_t ulInterfaceCnt = 1;
  uint32_t ulBoard        = 0;

  EnterCriticalSection(&m_tcsInterfaceLock);

  INTERFACE_MAP::iterator iter = m_cmInterfaces.begin();

  while( iter != m_cmInterfaces.end())
  {
    INTERFACE_DATA_T& tInterface      = (iter++)->second;
    std::string       szInterfaceName;
    BOOL              fInterfaceProgress = FALSE;
    int32_t           lRet = CIFX_NO_ERROR;


    // Return the complete name for the user
    tInterface.pcInterface->GetInterfaceName(szInterfaceName);

    /* Check if we need to start the interface */
    tInterface.pcInterface->Lock();
    LeaveCriticalSection(&m_tcsInterfaceLock);

    if( eOFFLINE == tInterface.eState )
    {
      /* Interface offline or paused. Try to establish a connection before finding the board */
      lRet = EstablishConnection(tInterface);
    }
    tInterface.pcInterface->Unlock();

    if( CIFX_NO_ERROR == lRet)
    {
      /* We are able to establish the connection to the interface, so we can try to also open endpoints on it */
      for(size_t uiEndpoint = 0; uiEndpoint < tInterface.cvEndpoints.size(); uiEndpoint++)
      {
        CEndpoint*  pcEndpoint        = tInterface.cvEndpoints[uiEndpoint];
        uint32_t    ulEndpointBoardNr = 0;
        CDevice*    pcDevice          = NULL;

        if(pcEndpoint->m_pcDefaultDataLayer == NULL)
          continue;

        while ( NULL != (pcDevice = pcEndpoint->GetDeviceObject(ulEndpointBoardNr)))
        {
          BOARD_INFORMATION tBoardInfo = pcDevice->m_tBoardInfo;
          CreateInterfaceName(szInterfaceName,
                              pcDevice->m_tBoardInfo.abBoardName,
                              sizeof(pcDevice->m_tBoardInfo.abBoardName)+sizeof(pcDevice->m_tBoardInfo.abBoardAlias),
                              tBoardInfo.abBoardName);
          if (pfnCallback)
          {
            pfnCallback(ulBoard, &tBoardInfo, ulInterfaceCnt, (uint32_t)m_cmInterfaces.size(), pvUser, FALSE, NXAPI_NO_ERROR);
            fInterfaceProgress = TRUE;
          }
          cvBoardList.push_back(tBoardInfo);
          ulBoard++;
          ulEndpointBoardNr++;
        }
      }
    }

    /* Signal progress of interface enumeration */
    if (pfnCallback && !fInterfaceProgress)
    {
      BOARD_INFORMATION tBoardInfo;
      strcpy(tBoardInfo.abBoardName, szInterfaceName.c_str());
      pfnCallback(0, &tBoardInfo, ulInterfaceCnt, (uint32_t)m_cmInterfaces.size(), pvUser, FALSE, NXAPI_NO_ENTRY_FOUND);
    }

    ulInterfaceCnt++;
    EnterCriticalSection(&m_tcsInterfaceLock);
  }

  /* Signal that we are finishd with device browsing */
  if (pfnCallback)
  {
    pfnCallback(0, NULL, (uint32_t)m_cmInterfaces.size(), (uint32_t)m_cmInterfaces.size(), pvUser, TRUE, NXAPI_NO_ENTRY_FOUND);
  }

  LeaveCriticalSection(&m_tcsInterfaceLock);

  return lRet;
}

///////////////////////////////////////////////////////////////////////////////////////////
/// Enuerate available channels on a given device
///   \param  ulBoard       Board number
///   \param  ulChannel     Board number
///   \param  ptChannelInfo Buffer for the channel information
///   \return  CIFX_NO_ERROR on success
///////////////////////////////////////////////////////////////////////////////////////////
int32_t CDeviceHandler::EnumerateChannels(uint32_t ulBoard, uint32_t ulChannel, CHANNEL_INFORMATION* ptChannelInfo)
{
  int32_t     lRet        = CIFX_NO_MORE_ENTRIES;
  CDevice* pcDevice    = GetDevice(ulBoard);

  if(NULL == pcDevice)
  {
    lRet = CIFX_INVALID_BOARD;
  } else
  {
    std::string szInterfaceName;
    pcDevice->m_pcEndpoint->m_pcInterface->GetInterfaceName(szInterfaceName);

    if( CIFX_NO_ERROR == (lRet = pcDevice->EnumChannel(ulChannel, ptChannelInfo)) )
    {
      // Return the complete name for the user
      CreateInterfaceName(szInterfaceName,
                          pcDevice->m_tBoardInfo.abBoardName,
                          sizeof(ptChannelInfo->abBoardName)+sizeof(ptChannelInfo->abBoardAlias),
                          ptChannelInfo->abBoardName);
    }
  }
  return lRet;
}

///////////////////////////////////////////////////////////////////////////////////////////
/// Create the interface name for the user
///   \param  szInterfaceName   Name of the physical interface
///   \param  pszEnpointName    Name of the device
///   \param  ulNameBufferSize  Size of the name buffer
///   \param  pszNameBuffer     Resulting name
///   \return  CIFX_NO_ERROR on success
///////////////////////////////////////////////////////////////////////////////////////////
void CDeviceHandler::CreateInterfaceName(std::string&   szInterfaceName,
                                         char*          pszEnpointName,
                                         uint32_t  ulNameBufferSize,
                                         char*          pszNameBuffer)
{
  _snprintf(pszNameBuffer, ulNameBufferSize,"%s_%s", szInterfaceName.c_str(), pszEnpointName);
}

///////////////////////////////////////////////////////////////////////////////////////////
/// Create the interface name for the user
///   \param  eResetCmd     Reset command to handle
///   \param  pcChannel     Channel object
///   \param  ulTimeout     Function timeout
///   \param  ulMode        Reset mode and parameter
///   \return  CIFX_NO_ERROR on success
///////////////////////////////////////////////////////////////////////////////////////////
int32_t CDeviceHandler::HandleReset(DEVICE_HANDLER_RESET_CMD_E eResetCmd,
                                    CChannel*                  pcChannel,
                                    uint32_t                   ulTimeout,
                                    uint32_t                   ulMode)
{
  int32_t lRet = CIFX_NO_ERROR;

  CDevice*            pcDevCon         = pcChannel->m_pcDevice;
  CDataLayer*         pcDataLayer      = pcDevCon->m_pcEndpoint->m_pcDefaultDataLayer;
  CPhysicalInterface* pcInterface      = pcDevCon->m_pcEndpoint->m_pcInterface;
  bool                fExpectReconnect;

  if(pcDataLayer->GetTransportLayer()->GetServerInfo()->ulFeatures & HIL_TRANSPORT_FEATURES_PERMANENT_CONNECTION)
  {
    fExpectReconnect = false;
  } else
  {
    fExpectReconnect = true;
  }

  /* Query interface name */
  std::string      szInterface;
  pcInterface->GetInterfaceName(szInterface);

  TRACE("%s: Reset...", szInterface.c_str());

  /* Tell Device handler to notify us, if Interface has reconnected */
  EnterCriticalSection(&m_tcsInterfaceLock);
  /* Search if we already know this Interface */
  INTERFACE_DATA_T& tInterface = m_cmInterfaces.find(szInterface)->second;
  pcInterface->m_hReconnect = ::CreateEvent(NULL, FALSE, FALSE, NULL);
  LeaveCriticalSection(&m_tcsInterfaceLock);

  /* Query interface specific reset timeout */
  uint32_t ulIntfResetTimeout = pcInterface->GetResetTimeout();

  /* Use interface specific reset timeout if no timeout value is given (zero)
     or given timeout is less than interface specific reset timeout           */
  if ( (0                  == ulTimeout) ||
       (ulIntfResetTimeout >  ulTimeout)   )
  {
    ulTimeout = ulIntfResetTimeout;
  }

  ScheduleInterfaceMonitor(szInterface, eRESET);

  switch(eResetCmd)
  {
    case eSYSTEMRESET:
      lRet = pcDataLayer->xSysdeviceReset(pcChannel->m_hChannel,
                                          CIFX_TO_SEND_PACKET);
    break;

    case eSYSTEMRESETEX:
      lRet = pcDataLayer->xSysdeviceResetEx(pcChannel->m_hChannel,
                                            ulTimeout,
                                            ulMode);
    break;

    case eCHANNELRESET:
      lRet = pcDataLayer->xChannelReset(pcChannel->m_hChannel,
                                        CIFX_SYSTEMSTART,
                                        CIFX_TO_SEND_PACKET);
    break;

    case eCHANNELINIT:
      lRet = pcDataLayer->xChannelReset(pcChannel->m_hChannel,
                                        CIFX_CHANNELINIT,
                                        CIFX_TO_SEND_PACKET);
      fExpectReconnect = false;
    break;

    default:
      lRet = CIFX_INVALID_COMMAND;
    break;

  }

  /* ATTENTION: Some devices drop connection on channelinit too (if remote device is TCP/IP
                based and the channel being reset also reset TCP/IP stack) */
  if( fExpectReconnect                      ||
      (CIFX_TRANSPORT_ABORTED      == lRet) ||
      (CIFX_TRANSPORT_RECV_TIMEOUT == lRet) )
  {
    /* No Answer returned, so we expect to reconnect. Use interface specific reset timeout */
    if(::WaitForSingleObject(pcInterface->m_hReconnect, ulTimeout) != WAIT_OBJECT_0)
    {
      /* we need to reconnect manually */
      tInterface.pcInterface->Lock();
      InterfaceDisconnect(tInterface);
      TRACE("%s: Reset timeout expired..disconnect from device (Error: 0x%x)", szInterface.c_str(), lRet);

      lRet = EstablishConnection(tInterface);
      TRACE("%s: Reset timeout expired..connect to device (Error: 0x%x)", szInterface.c_str(), lRet);
      tInterface.pcInterface->Unlock();
    }
  }

  if(!pcDevCon->IsAvailable())
  {
    lRet = CIFX_DEV_RESET_TIMEOUT;
    TRACE("%s: Reset failed! Timeout (%d sec)", szInterface.c_str(), ulTimeout);
  }
  else
  {
    lRet = CIFX_NO_ERROR;
    TRACE("%s: Reset done", szInterface.c_str());
  }

  /* Unregister from reconnect events */
  ::CloseHandle(pcInterface->m_hReconnect);
  pcInterface->m_hReconnect = NULL;

  return lRet;
}

///////////////////////////////////////////////////////////////////////////////////////////
/// Enumerate a channel by a given channel number
///   \param  ulChannel     Requested channel numbe
///   \param  ptChannelInfo Buffer to store the channel information
///   \return CIFX_NO_ERROR on success
///////////////////////////////////////////////////////////////////////////////////////////
int32_t CDevice::EnumChannel(uint32_t ulChannel, CHANNEL_INFORMATION* ptChannelInfo)
{
  int32_t lRet = CIFX_NO_MORE_ENTRIES;

  std::map<uint32_t, CChannel*>::iterator iter = m_cmChannels.find(ulChannel);

  if(iter != m_cmChannels.end())
  {
    CChannel* pcChannel = iter->second;

    *ptChannelInfo = pcChannel->m_tChannelInfo;
    lRet = CIFX_NO_ERROR;
  }

  return lRet;
}

///////////////////////////////////////////////////////////////////////////////////////////
/// Destructor of the CEndpoint class
///////////////////////////////////////////////////////////////////////////////////////////
CEndpoint::~CEndpoint()
{
  /* Remove devices and channels */
  while(m_cvDevices.size() > 0)
  {
    CDevice* pcDevice = *m_cvDevices.begin();
    delete pcDevice;

    m_cvDevices.erase(m_cvDevices.begin());
  }

  /* Deinit and remove data layers */
  while(m_cmDataLayers.size() > 0)
  {
    std::map<uint16_t, CDataLayer*>::iterator iter = m_cmDataLayers.begin();
    CDataLayer*                    pcDataLayer   = iter->second;
    pcDataLayer->Deinit();
    delete pcDataLayer;

    m_cmDataLayers.erase(m_cmDataLayers.begin());
  }
}

/////////////////////////////////////////////////////////////////////////////
/// Get a device object by a given board number
///   \param  ulBoard   Board number
///   \return CDevice* device object
/////////////////////////////////////////////////////////////////////////////
CDevice*  CEndpoint::GetDeviceObject(uint32_t ulBoard)
{
  CDevice* pcRet = NULL;

  if(ulBoard < m_cvDevices.size())
  {
    pcRet = m_cvDevices[ulBoard];
  }

  return pcRet;
}

/////////////////////////////////////////////////////////////////////////////
/// Get a channel object from a device by a given channel number
///   \param  ulChannel Channel number
///   \return CChannel* channel object
/////////////////////////////////////////////////////////////////////////////
CChannel* CDevice::GetChannelObject(uint32_t ulChannel)
{
  CChannel* pcRet = NULL;

  std::map<uint32_t, CChannel*>::iterator iter = m_cmChannels.find(ulChannel);

  if(iter != m_cmChannels.end())
  {
    pcRet = iter->second;
  }

  return pcRet;
}

/////////////////////////////////////////////////////////////////////////////
/// Get a device object by a given device name
///   \param  szBoard   Device name string
///   \return CDevice* channel object
/////////////////////////////////////////////////////////////////////////////
CDevice* CEndpoint::GetDeviceObject(const char* szBoard)
{
  CDevice* pcRet = NULL;

  for(size_t uiBoard = 0; uiBoard < m_cvDevices.size(); uiBoard++)
  {
    CDevice* pcTempDevice = m_cvDevices[uiBoard];

    if(0 == strncmp((char*)pcTempDevice->m_tBoardInfo.abBoardName,
                    szBoard,
                    sizeof(pcTempDevice->m_tBoardInfo.abBoardName)))
    {
      pcRet = pcTempDevice;
      break;
    }
  }

  return pcRet;
}

/////////////////////////////////////////////////////////////////////////////
/// Add a device to the device map
///   \param  tBoardInfo          Board information structure
///   \return CDevice* device handle
/////////////////////////////////////////////////////////////////////////////
CDevice*  CEndpoint::AddDevice(BOARD_INFORMATION& tBoardInfo)
{
  CDevice* pcRet = NULL;

  /* Search for this Device */
  for(size_t uiIdx = 0; uiIdx < m_cvDevices.size(); uiIdx++)
  {
    CDevice* pcTemp = m_cvDevices[uiIdx];

    if(strcmp(pcTemp->m_tBoardInfo.abBoardName,
              tBoardInfo.abBoardName) == 0)
    {
      /* We've found the device */
      pcRet = pcTemp;
      /* Overwrite existing board information */
      pcRet->m_tBoardInfo = tBoardInfo;
      break;
    }
  }

  if(NULL == pcRet)
  {
    /* We did not find it, so allocate a new device */
    pcRet = new CDevice(this);
    pcRet->m_tBoardInfo = tBoardInfo;

    m_cvDevices.push_back(pcRet);
  }

  pcRet->SetAvailable(true);

  return pcRet;
}

/////////////////////////////////////////////////////////////////////////////
/// Return number of devices registered on endpoint
///   \return Number of devices
/////////////////////////////////////////////////////////////////////////////
uint32_t CEndpoint::GetDeviceCount()
{
  return (uint32_t) m_cvDevices.size();
}

/////////////////////////////////////////////////////////////////////////////
/// Add a channel to the channel map
///   \param  ulChannel     Channel number
///   \param  ptChannelInfo Channel information structure
///   \return CChannel* channel handle
/////////////////////////////////////////////////////////////////////////////
CChannel* CDevice::AddChannel(uint32_t ulChannel, CHANNEL_INFORMATION* ptChannelInfo)
{
  CChannel* pcRet = NULL;

  std::map<uint32_t,CChannel*>::iterator iter = m_cmChannels.find(ulChannel);

  if(iter != m_cmChannels.end())
  {
    /* We've found the channel */
    pcRet = iter->second;

    /* TODO: How do we verify that we are using the correct
             Channelinformation if there are multiple
             Datalayers available?? */
    if(NULL != ptChannelInfo)
      pcRet->m_tChannelInfo = *ptChannelInfo;

  } else
  {
    pcRet = new CChannel(this, ulChannel, NULL);

    if(NULL != ptChannelInfo)
      pcRet->m_tChannelInfo = *ptChannelInfo;

    m_cmChannels[ulChannel] = pcRet;
  }

  pcRet->SetAvailable(true);

  return pcRet;
}

/////////////////////////////////////////////////////////////////////////////
/// Interface monitor thread
///   \param  pvParam   User data
///   \return always 0
/////////////////////////////////////////////////////////////////////////////
DWORD CDeviceHandler::InterfaceMonitorThread(void* pvParam)
{
  CDeviceHandler* pcDevice = reinterpret_cast<CDeviceHandler*>(pvParam);

  pcDevice->InterfaceMonitorThreadFunc();

  return 0;
}

/////////////////////////////////////////////////////////////////////////////
/// Interface monitor thread function
/////////////////////////////////////////////////////////////////////////////
void CDeviceHandler::InterfaceMonitorThreadFunc(void)
{
  bool    fRunning = true;

  do
  {
    DWORD dwWait = ::WaitForSingleObject(m_hInterfaceMonitorStop, MONITOR_THREAD_TIMEOUT);

    switch(dwWait)
    {
    case WAIT_OBJECT_0:
      fRunning = false;
      break;

    case WAIT_TIMEOUT:
      {
        uint32_t ulCurTime = GetTickCount();
        std::map<std::string,INTERFACE_MONITOR_COMMAND_E> cmProcess;

        EnterCriticalSection(&m_tcsInterfaceMonitorQueue);
        INTERFACE_MONITOR_QUEUE::iterator iter = m_cmInterfaceMonitorQueue.begin();
        while (iter != m_cmInterfaceMonitorQueue.end())
        {
          INTERFACE_MONITOR_DATA_T* ptMonitorData = &iter->second;
          if (ptMonitorData->ulTimer <= ulCurTime)
          {
            cmProcess.insert(make_pair(iter->first, ptMonitorData->eCmd));
            m_cmInterfaceMonitorQueue.erase(iter);
            iter = m_cmInterfaceMonitorQueue.begin();
          } else
          {
            iter++;
          }
        }
        LeaveCriticalSection(&m_tcsInterfaceMonitorQueue);

        std::map<std::string,INTERFACE_MONITOR_COMMAND_E>::iterator iterProcess = cmProcess.begin();
        while (iterProcess != cmProcess.end())
        {
          INTERFACE_DATA_T& tInterface = m_cmInterfaces.find(iterProcess->first)->second;
          if (iterProcess->second == eRECONNECT)
          {
            tInterface.pcInterface->Lock();

            InterfaceDisconnect(tInterface);
            int32_t lError = EstablishConnection(tInterface);

            tInterface.pcInterface->Unlock();
            if(CIFX_NO_ERROR != lError)
            {
              TRACE("%s: Reconnect failed (Error: 0x%x)", tInterface.pcInterface->GetInterfaceName().c_str(), lError);
              /* Reconnect failed, retry reconnect */
              ScheduleInterfaceMonitor(tInterface.pcInterface->GetInterfaceName(), eRECONNECT);
            } else
            {
              TRACE("%s: Reconnect suceeded!", tInterface.pcInterface->GetInterfaceName().c_str());
            }
          } else
          {
            bool fAccessed = false;
            tInterface.pcInterface->Lock();
            for(std::vector<CEndpoint*>::iterator iterEndpoint = tInterface.cvEndpoints.begin(); iterEndpoint != tInterface.cvEndpoints.end(); iterEndpoint++)
            {
              uint32_t      ulDevice   = 0;
              CDevice*   pcDevice   = NULL;
              while(NULL != (pcDevice = (*iterEndpoint)->GetDeviceObject(ulDevice++)))
              {
                if (pcDevice->IsAccessed())
                {
                  fAccessed = true;
                  break;
                }
              }
              if (fAccessed) break;
            }
            if (!fAccessed)
            {
              InterfaceDisconnect(tInterface);
              tInterface.eState = ePAUSED;
            }
            tInterface.pcInterface->Unlock();
          }
          iterProcess++;
        }

      }
      break;

    default:
      _ASSERT(false);
      break;
    }

  } while(fRunning);
}

/////////////////////////////////////////////////////////////////////////////
/// Schedule monitor to reconnect or disconnect an interface
///   \param szInterface Returns interface name associated with the given endpoint
///   \param eCmd        eDISCONNECT/eRECONNECT/eRESET
/////////////////////////////////////////////////////////////////////////////
void CDeviceHandler::ScheduleInterfaceMonitor(std::string& szInterface, INTERFACE_MONITOR_COMMAND_E eCmd)
{
  EnterCriticalSection(&m_tcsInterfaceMonitorQueue);
  INTERFACE_MONITOR_QUEUE::iterator iter = m_cmInterfaceMonitorQueue.find(szInterface);
  if(iter == m_cmInterfaceMonitorQueue.end())
  {
    switch (eCmd)
    {
      case eDISCONNECT:
        {
          INTERFACE_MONITOR_DATA_T tMonitorData;
          tMonitorData.ulTimer = GetTickCount() + MONITOR_DISCONNECT_TIMEOUT;
          tMonitorData.eCmd    = eDISCONNECT;
          m_cmInterfaceMonitorQueue.insert(make_pair(szInterface, tMonitorData));
          TRACE("%s: Schedule disconnect", szInterface.c_str());
        }
        break;

      case eRECONNECT:
        {
          INTERFACE_MONITOR_DATA_T tMonitorData;
          tMonitorData.ulTimer = GetTickCount() + MONITOR_RECONNECT_TIMEOUT;
          tMonitorData.eCmd    = eRECONNECT;
          m_cmInterfaceMonitorQueue.insert(make_pair(szInterface, tMonitorData));
          TRACE("%s: Schedule reconnect", szInterface.c_str());
        }
        break;

      default:
        break;
    }
  } else
  {
    switch (eCmd)
    {
      case eDISCONNECT:
        {
          INTERFACE_MONITOR_DATA_T* ptMonitorData = &iter->second;
          if (ptMonitorData->eCmd == eDISCONNECT)
          {
            ptMonitorData->ulTimer = GetTickCount() + MONITOR_DISCONNECT_TIMEOUT;
          }
        }
        break;

      case eRESET:
        m_cmInterfaceMonitorQueue.erase(iter);
        TRACE("%s: Reset Monitor", szInterface.c_str());
        break;

      default:
        break;
    }
  }

  LeaveCriticalSection(&m_tcsInterfaceMonitorQueue);
}

/////////////////////////////////////////////////////////////////////////////
/// Check interface connections
///   \param tInterface Interface data
///   \param fForce     Disconnect even if interface is still accessed
/////////////////////////////////////////////////////////////////////////////
void CDeviceHandler::InterfaceDisconnect(INTERFACE_DATA_T& tInterface)
{
  tInterface.pcInterface->Lock();
  if (tInterface.eState == eONLINE)
  {
    // Stop physical interface
    tInterface.pcInterface->Stop();
    TRACE("%s: Disconnect", tInterface.pcInterface->GetInterfaceName().c_str());
    // Remove data layer
    for( uint32_t ulIdx = 0; ulIdx < tInterface.cvEndpoints.size(); ulIdx++)
    {
      CEndpoint* pcEndpoint = tInterface.cvEndpoints[ulIdx];
      uint32_t      ulDevice   = 0;
      CDevice*   pcDevice   = NULL;

      /* Mark device and channels not available */
      while(NULL != (pcDevice = pcEndpoint->GetDeviceObject(ulDevice++)))
      {
        pcDevice->SetAvailable(false);
      }

      /* Uninitialize all data layers */
      std::map<uint16_t, CDataLayer*>::iterator iterDataLayer = pcEndpoint->m_cmDataLayers.begin();

      while(iterDataLayer != pcEndpoint->m_cmDataLayers.end())
      {
        CDataLayer* pcDataLayer = iterDataLayer->second;
        pcDataLayer->Deinit();
        iterDataLayer++;
      }
    }

    // Stop transport layer
    tInterface.pcInterface->GetTransportLayer()->DeInit();
  }
  tInterface.eState = eOFFLINE;
  tInterface.pcInterface->Unlock();
}

/////////////////////////////////////////////////////////////////////////////
/// Split device name into interface name and physical device name (e.g. TCP0_cifX0)
///   \param szFullName   Full name given to the user
///   \param szInterface  Buffer for the interface name
///   \param szBoard      Buffer for the device/board name
///   \return true on success
/////////////////////////////////////////////////////////////////////////////
bool CDeviceHandler::SplitDeviceName(char* szFullName, std::string& szInterface, std::string& szBoard)
{
  bool fRet = false;

  /* Splitup I/F / Board */
  std::string   cFullName   = szFullName;
  int           iBoardStart = (int)cFullName.find("_");

  szInterface.clear();
  szBoard.clear();

  if(-1 != iBoardStart)
  {
    szInterface.append(cFullName.begin(), cFullName.begin() + iBoardStart);
    szBoard.append(cFullName.begin() + iBoardStart + 1, cFullName.end());

    fRet = true;
  }

  return fRet;
}

/////////////////////////////////////////////////////////////////////////////
/// Get physical layer via connector identifier prefix
///   \param szLayerPrefix  Connector identifier prefix to find
///   \return Pointer reference to physical layer
/////////////////////////////////////////////////////////////////////////////
CPhysicalLayer* CDeviceHandler::GetLayer( const char* szLayerPrefix)
{
  CPhysicalLayer* pcRet = NULL;

  for(size_t uiIdx = 0; uiIdx < m_cvLayers.size(); uiIdx++)
  {
    const char* szConPrefix = m_cvLayers[uiIdx]->GetConnectorInfo()->szConIdentifier;

    if(0 == strncmp(szConPrefix, szLayerPrefix, strlen(szConPrefix)))
    {
      pcRet = m_cvLayers[uiIdx];
      break;
    }
  }

  return pcRet;
}

/////////////////////////////////////////////////////////////////////////////
/// Get physical layer via index
///   \param ulIdx Index
///   \return Pointer reference to physical layer
/////////////////////////////////////////////////////////////////////////////
CPhysicalLayer* CDeviceHandler::GetLayer(uint32_t ulIdx)
{
  CPhysicalLayer* pcRet = NULL;

  if(ulIdx < m_cvLayers.size())
  {
    pcRet = m_cvLayers[ulIdx];
  }

  return pcRet;
}
